@extends('admin.layout')

@section('title', 'Produtos')
@section('page_title', 'Gestão de Produtos')

@section('content')
<div class="mb-6 flex justify-between items-center">
    <div></div>
    <button onclick="openModal()" class="btn-primary">
        <i class="fas fa-plus mr-2"></i> Novo Produto
    </button>
</div>

<div class="bg-white rounded shadow overflow-x-auto">
    <table class="w-full">
        <thead class="bg-gray-50 border-b">
            <tr>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Título</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Categoria</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Preço</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Stock</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Imagens</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Oferta Dia</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Combo</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ativo</th>
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ações</th>
            </tr>
        </thead>
        <tbody class="divide-y divide-gray-200">
            @forelse($products as $product)
                <tr class="hover:bg-gray-50">
                    <td class="px-6 py-4 font-semibold">{{ $product->title }}</td>
                    <td class="px-6 py-4">{{ $product->category->name ?? 'N/A' }}</td>
                    <td class="px-6 py-4 font-bold">{{ number_format($product->price, 2) }} KZ</td>
                    <td class="px-6 py-4">{{ $product->stock }}</td>
                    <td class="px-6 py-4">
                        @if($product->images && count($product->images) > 0)
                            <span class="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded">{{ count($product->images) }} imagens</span>
                        @else
                            <span class="px-2 py-1 bg-gray-100 text-gray-600 text-xs rounded">Sem imagens</span>
                        @endif
                    </td>
                    <td class="px-6 py-4">
                        <button onclick="toggleDailyDeal({{ $product->id }}, {{ json_encode($product) }})" class="px-3 py-1 rounded text-xs font-semibold transition-colors {{ $product->is_daily_deal ? 'bg-orange-500 text-white hover:bg-orange-600' : 'bg-gray-300 text-gray-800 hover:bg-gray-400' }}">
                            {{ $product->is_daily_deal ? '✓ Oferta' : 'Oferta' }}
                        </button>
                    </td>
                    <td class="px-6 py-4">
                        <button onclick="toggleCombo({{ $product->id }}, {{ json_encode($product) }})" class="px-3 py-1 rounded text-xs font-semibold transition-colors {{ $product->is_combo ? 'bg-teal-500 text-white hover:bg-teal-600' : 'bg-gray-300 text-gray-800 hover:bg-gray-400' }}">
                            {{ $product->is_combo ? '✓ Combo' : 'Combo' }}
                        </button>
                    </td>
                    <td class="px-6 py-4">
                        <span class="px-2 py-1 rounded text-white text-xs {{ $product->active ? 'bg-green-500' : 'bg-gray-400' }}">
                            {{ $product->active ? 'Sim' : 'Não' }}
                        </span>
                    </td>
                    <td class="px-6 py-4 flex gap-2">
                        <button onclick="editProduct({{ $product->id }}, {{ json_encode($product) }})" class="btn-edit">
                            <i class="fas fa-edit"></i> Editar
                        </button>
                        <form action="{{ route('admin.products.destroy', $product) }}" method="POST" class="inline">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="btn-danger" onclick="return confirm('Tem a certeza?')">
                                <i class="fas fa-trash"></i> Remover
                            </button>
                        </form>
                    </td>
                </tr>
            @empty
                <tr>
                    <td colspan="7" class="px-6 py-4 text-center text-gray-500">Sem produtos</td>
                </tr>
            @endforelse
        </tbody>
    </table>
</div>

<!-- Modal -->
<div id="productModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded shadow-lg p-8 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <h2 class="text-2xl font-bold mb-6">Novo Produto</h2>
        <form id="productForm" method="POST" enctype="multipart/form-data">
            @csrf
            <div class="grid grid-cols-2 gap-4 mb-4">
                <div>
                    <label class="block text-gray-700 text-sm font-bold mb-2">Título</label>
                    <input type="text" name="title" class="w-full px-4 py-2 border rounded" required>
                </div>
                <div>
                    <label class="block text-gray-700 text-sm font-bold mb-2">Categoria</label>
                    <select name="category_id" class="w-full px-4 py-2 border rounded" required>
                        <option value="">Selecione</option>
                        @foreach($categories as $category)
                            <option value="{{ $category->id }}">{{ $category->name }}</option>
                        @endforeach
                    </select>
                </div>
            </div>

            <div class="grid grid-cols-3 gap-4 mb-4">
                <div>
                    <label class="block text-gray-700 text-sm font-bold mb-2">Preço</label>
                    <input type="number" name="price" step="0.01" class="w-full px-4 py-2 border rounded" required>
                </div>
                <div>
                    <label class="block text-gray-700 text-sm font-bold mb-2">Preço Original</label>
                    <input type="number" name="original_price" step="0.01" class="w-full px-4 py-2 border rounded">
                </div>
                <div>
                    <label class="block text-gray-700 text-sm font-bold mb-2">Stock</label>
                    <input type="number" name="stock" class="w-full px-4 py-2 border rounded" required>
                </div>
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-bold mb-2">Descrição</label>
                <textarea name="description" class="w-full px-4 py-2 border rounded" rows="3"></textarea>
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-bold mb-2">Imagens (Upload)</label>
                <div id="imagesContainer">
                    <div class="image-input-group mb-2">
                        <input type="file" class="image-file w-full px-4 py-2 border rounded" accept="image/*">
                    </div>
                </div>
                <button type="button" onclick="addImageInput()" class="text-blue-600 hover:text-blue-800 text-sm font-semibold mt-2">
                    <i class="fas fa-plus"></i> Adicionar outra imagem
                </button>
                <div id="imagesPreview" class="mt-4 grid grid-cols-2 gap-4"></div>
            </div>

            <div class="mb-4 grid grid-cols-2 gap-4">
                <label class="flex items-center">
                    <input type="checkbox" name="is_daily_deal" value="1" class="mr-2">
                    <span class="text-gray-700">Oferta do Dia</span>
                </label>
                <label class="flex items-center">
                    <input type="checkbox" name="is_combo" value="1" class="mr-2">
                    <span class="text-gray-700">Combo Especial</span>
                </label>
            </div>

            <div class="mb-4">
                <label class="flex items-center">
                    <input type="checkbox" name="active" value="1" class="mr-2" checked>
                    <span class="text-gray-700">Ativo</span>
                </label>
            </div>

            <div class="flex gap-4">
                <button type="submit" class="btn-primary flex-1">Salvar Produto</button>
                <button type="button" onclick="closeModal()" class="flex-1 px-4 py-2 border border-gray-300 rounded hover:bg-gray-50">Cancelar</button>
            </div>
        </form>
    </div>
</div>

<script>
// Store images to remove
let imagesToRemove = [];

function openModal() {
    imagesToRemove = [];
    document.getElementById('productModal').classList.remove('hidden');
    document.getElementById('productForm').action = "{{ route('admin.products.store') }}";
    document.getElementById('productForm').method = 'POST';
    document.getElementById('imagesPreview').innerHTML = '';
    document.getElementById('productForm').reset();
    document.getElementById('imagesContainer').innerHTML = '<div class="image-input-group mb-2"><input type="file" class="image-file w-full px-4 py-2 border rounded" accept="image/*"></div>';
    document.querySelectorAll('.image-file').forEach(input => input.addEventListener('change', updatePreview));
}

function closeModal() {
    imagesToRemove = [];
    document.getElementById('productModal').classList.add('hidden');
}

function addImageInput() {
    const container = document.getElementById('imagesContainer');
    const div = document.createElement('div');
    div.className = 'image-input-group mb-2';
    div.innerHTML = `
        <div class="flex gap-2">
            <input type="file" class="image-file flex-1 px-4 py-2 border rounded" accept="image/*">
            <button type="button" onclick="this.parentElement.parentElement.remove()" class="btn-danger">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    container.appendChild(div);
    
    // Add change listener to new input
    const fileInput = div.querySelector('.image-file');
    fileInput.addEventListener('change', updatePreview);
}

function updatePreview() {
    const preview = document.getElementById('imagesPreview');
    preview.innerHTML = '';
    
    const fileInputs = document.querySelectorAll('.image-file');
    fileInputs.forEach((input, index) => {
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                const div = document.createElement('div');
                div.className = 'relative';
                div.innerHTML = `
                    <img src="${e.target.result}" alt="Preview ${index}" class="w-full h-32 object-cover rounded">
                    <p class="text-xs text-gray-600 mt-1">${input.files[0].name}</p>
                `;
                preview.appendChild(div);
            };
            reader.readAsDataURL(input.files[0]);
        }
    });
}

function editProduct(id, product) {
    imagesToRemove = [];
    document.getElementById('productModal').classList.remove('hidden');
    document.getElementById('productForm').action = `/admin/products/${id}`;
    document.getElementById('productForm').method = 'POST';
    
    // Pré-preencher dados
    document.querySelector('input[name="title"]').value = product.title;
    document.querySelector('select[name="category_id"]').value = product.category_id;
    document.querySelector('input[name="price"]').value = product.price;
    document.querySelector('input[name="original_price"]').value = product.original_price || '';
    document.querySelector('input[name="stock"]').value = product.stock;
    document.querySelector('textarea[name="description"]').value = product.description || '';
    document.querySelector('input[name="is_daily_deal"]').checked = product.is_daily_deal || false;
    document.querySelector('input[name="is_combo"]').checked = product.is_combo || false;
    document.querySelector('input[name="active"]').checked = product.active;
    
    // Mostrar imagens existentes
    const preview = document.getElementById('imagesPreview');
    preview.innerHTML = '';
    if (product.images && Array.isArray(product.images)) {
        product.images.forEach((img, index) => {
            const div = document.createElement('div');
            div.className = 'relative';
            div.innerHTML = `
                <img src="${img}" alt="Imagem ${index}" class="w-full h-32 object-cover rounded">
                <button type="button" class="absolute top-1 right-1 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600" onclick="removeExistingImage('${img}', this)">✕</button>
            `;
            preview.appendChild(div);
        });
    }
    
    // Limpar inputs de arquivo
    document.getElementById('imagesContainer').innerHTML = '<div class="image-input-group mb-2"><input type="file" class="image-file w-full px-4 py-2 border rounded" accept="image/*"></div>';
    document.querySelectorAll('.image-file').forEach(input => input.addEventListener('change', updatePreview));
    
    // Adicionar método PATCH
    const form = document.getElementById('productForm');
    const methodInput = form.querySelector('input[name="_method"]');
    if (methodInput) {
        methodInput.value = 'PATCH';
    } else {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = '_method';
        input.value = 'PATCH';
        form.appendChild(input);
    }
}

function removeExistingImage(imagePath, btn) {
    // Adicionar à lista de imagens a remover
    if (!imagesToRemove.includes(imagePath)) {
        imagesToRemove.push(imagePath);
    }
    
    // Remover do visual
    btn.parentElement.remove();
}

document.querySelectorAll('.image-file').forEach(input => {
    input.addEventListener('change', updatePreview);
});

function toggleDailyDeal(id, product) {
    if (confirm(product.is_daily_deal ? 'Remover de Oferta do Dia?' : 'Marcar como Oferta do Dia?')) {
        const formData = new FormData();
        formData.append('is_daily_deal', product.is_daily_deal ? '0' : '1');
        formData.append('_method', 'PATCH');
        
        fetch(`/admin/products/${id}`, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => {
            if (response.ok) {
                window.location.reload();
            } else {
                alert('Erro ao atualizar');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro ao atualizar');
        });
    }
}

function toggleCombo(id, product) {
    if (confirm(product.is_combo ? 'Remover de Combo Especial?' : 'Marcar como Combo Especial?')) {
        const formData = new FormData();
        formData.append('is_combo', product.is_combo ? '0' : '1');
        formData.append('_method', 'PATCH');
        
        fetch(`/admin/products/${id}`, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => {
            if (response.ok) {
                window.location.reload();
            } else {
                alert('Erro ao atualizar');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro ao atualizar');
        });
    }
}

document.getElementById('productForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    // Remover images field anterior
    formData.delete('images');
    
    // Adicionar imagens a remover
    imagesToRemove.forEach(img => {
        formData.append('images_to_remove[]', img);
    });
    
    // Pegar todos os file inputs
    const fileInputs = document.querySelectorAll('.image-file');
    fileInputs.forEach((input, index) => {
        if (input.files && input.files[0]) {
            formData.append('image_files[]', input.files[0]);
        }
    });
    
    fetch(this.action, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => {
        if (response.ok) {
            imagesToRemove = [];
            window.location.reload();
        } else {
            alert('Erro ao salvar');
        }
    })
    .catch(error => console.error('Erro:', error));
});
</script>
@endsection
