<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SmsService
{
    private $apiKey;
    private $apiUrl;

    public function __construct()
    {
        $this->apiKey = env('TELCO_SMS_API_KEY', 'prd32a44c62fa0f801eea35169c14');
        $this->apiUrl = env('TELCO_SMS_API_URL', 'https://www.telcosms.co.ao/api/v2/send_message');
    }

    /**
     * Send SMS message
     */
    public function sendSms(string $phoneNumber, string $message): bool
    {
        try {
            Log::info('Attempting to send SMS', [
                'original_phone' => $phoneNumber,
                'message' => $message,
                'api_key' => substr($this->apiKey, 0, 10) . '...',
                'api_url' => $this->apiUrl
            ]);

            // Remove any non-numeric characters from phone number
            $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            // Ensure phone starts with country code or add it if needed
            if (!str_starts_with($cleanPhone, '244')) {
                $cleanPhone = '244' . $cleanPhone;
            }

            Log::info('Cleaned phone number', ['cleaned_phone' => $cleanPhone]);

            $payload = [
                'message' => [
                    'api_key_app' => $this->apiKey,
                    'phone_number' => $cleanPhone,
                    'message_body' => $message
                ]
            ];

            Log::info('Sending SMS request', ['payload' => $payload]);

            $response = Http::withOptions([
                    'verify' => false, // Desativar verificação SSL para TelcoSMS
                ])
                ->timeout(30)
                ->post($this->apiUrl, $payload);

            Log::info('SMS API Response', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            if ($response->successful()) {
                Log::info('SMS sent successfully', [
                    'phone' => $cleanPhone,
                    'response' => $response->json()
                ]);
                return true;
            } else {
                Log::error('SMS sending failed', [
                    'phone' => $cleanPhone,
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('SMS sending exception', [
                'phone' => $phoneNumber,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send order confirmation SMS
     */
    public function sendOrderConfirmation(string $phoneNumber, string $orderNumber, float $total): bool
    {
        $message = "SonaMarket: Pedido #{$orderNumber} confirmado! Total: " . number_format($total, 2, ',', '.') . " KZ. Obrigado por comprar conosco!";
        return $this->sendSms($phoneNumber, $message);
    }

    /**
     * Send order status update SMS
     */
    public function sendOrderStatusUpdate(string $phoneNumber, string $orderNumber, string $status): bool
    {
        $statusMessages = [
            'pending' => 'está pendente',
            'processing' => 'está em processamento',
            'shipped' => 'foi enviado e está a caminho',
            'delivered' => 'foi entregue',
            'cancelled' => 'foi cancelado'
        ];

        $statusText = $statusMessages[$status] ?? $status;
        $message = "SonaMarket: Seu pedido #{$orderNumber} {$statusText}. Rastreie: https://sonamarket.co.ao/tracking?order={$orderNumber}";
        
        return $this->sendSms($phoneNumber, $message);
    }

    /**
     * Send payment confirmation SMS
     */
    public function sendPaymentConfirmation(string $phoneNumber, string $orderNumber, string $paymentMethod): bool
    {
        $methodNames = [
            'express' => 'Express',
            'reference' => 'Referência',
            'transfer' => 'Transferência Bancária'
        ];

        $methodName = $methodNames[$paymentMethod] ?? $paymentMethod;
        $message = "SonaMarket: Pagamento confirmado para pedido #{$orderNumber} via {$methodName}. Obrigado!";
        
        return $this->sendSms($phoneNumber, $message);
    }
}
