<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    protected $fillable = [
        'category_id',
        'title',
        'slug',
        'description',
        'price',
        'original_price',
        'image',
        'images',
        'stock',
        'badge',
        'sold',
        'rating',
        'reviews',
        'active',
        'is_daily_deal',
        'is_combo',
    ];

    protected $casts = [
        'images' => 'array',
        'active' => 'boolean',
        'is_daily_deal' => 'boolean',
        'is_combo' => 'boolean',
        'price' => 'decimal:2',
        'original_price' => 'decimal:2',
        'rating' => 'decimal:1',
    ];

    public function getImageAttribute($value)
    {
        if (!$value) {
            return null;
        }
        
        // Se já é uma URL completa, retorna
        if (str_starts_with($value, 'http')) {
            return $value;
        }
        
        // Se é um caminho da API (imagens geradas dinamicamente)
        if (str_starts_with($value, '/api/')) {
            return 'http://localhost:8000' . $value;
        }
        
        // Se é um caminho de upload (começa com /uploads/), retorna URL completa do upload
        if (str_starts_with($value, '/uploads/')) {
            return 'http://localhost:8000' . $value;
        }
        
        // Se é um caminho de storage (não tem barra no início), adiciona storage
        if (!str_starts_with($value, '/')) {
            $value = '/' . $value;
        }
        
        return 'http://localhost:8000/storage' . $value;
    }
    
    protected function getImagesAttribute($value)
    {
        if (!$value) {
            return [];
        }
        
        // Se é string JSON, decodifica
        if (is_string($value)) {
            $images = json_decode($value, true) ?? [];
        } else {
            $images = $value;
        }
        
        // Converte cada imagem para URL completa
        return array_map(function($image) {
            if (!$image) {
                return null;
            }
            
            // Se já é uma URL completa, retorna
            if (str_starts_with($image, 'http')) {
                return $image;
            }
            
            // Se é um caminho de upload (começa com /uploads/), retorna URL completa
            if (str_starts_with($image, '/uploads/')) {
                return 'http://localhost:8000' . $image;
            }
            
            // Se é um caminho de storage
            if (!str_starts_with($image, '/')) {
                $image = '/' . $image;
            }
            
            return 'http://localhost:8000/storage' . $image;
        }, array_filter($images));
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }
}
