<?php

namespace App\Http\Controllers;

use App\Models\NewsletterSubscriber;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class NewsletterController extends Controller
{
    /**
     * Subscribe to newsletter
     */
    public function subscribe(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Email inválido',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Check if already subscribed
            $existing = NewsletterSubscriber::where('email', $request->email)->first();

            if ($existing) {
                // If previously unsubscribed, reactivate
                if (!$existing->is_active) {
                    $existing->update([
                        'is_active' => true,
                        'subscribed_at' => now(),
                        'unsubscribed_at' => null,
                        'ip_address' => $request->ip(),
                        'user_agent' => $request->userAgent(),
                    ]);

                    return response()->json([
                        'success' => true,
                        'message' => 'Inscrição reativada com sucesso!',
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Este email já está inscrito na newsletter.',
                ], 409);
            }

            // Create new subscription
            NewsletterSubscriber::create([
                'email' => $request->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'is_active' => true,
                'subscribed_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Inscrição realizada com sucesso! Obrigado por se inscrever.',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao processar inscrição. Tente novamente.',
            ], 500);
        }
    }

    /**
     * Get all subscribers (Admin only)
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 50);
        $status = $request->get('status', 'all'); // all, active, inactive

        $query = NewsletterSubscriber::query()->orderBy('created_at', 'desc');

        if ($status === 'active') {
            $query->active();
        } elseif ($status === 'inactive') {
            $query->inactive();
        }

        $subscribers = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $subscribers,
            'stats' => [
                'total' => NewsletterSubscriber::count(),
                'active' => NewsletterSubscriber::active()->count(),
                'inactive' => NewsletterSubscriber::inactive()->count(),
            ],
        ]);
    }

    /**
     * Get subscriber statistics
     */
    public function stats()
    {
        $stats = [
            'total' => NewsletterSubscriber::count(),
            'active' => NewsletterSubscriber::active()->count(),
            'inactive' => NewsletterSubscriber::inactive()->count(),
            'today' => NewsletterSubscriber::whereDate('created_at', today())->count(),
            'this_week' => NewsletterSubscriber::where('created_at', '>=', now()->startOfWeek())->count(),
            'this_month' => NewsletterSubscriber::whereMonth('created_at', now()->month)->count(),
        ];

        // Get subscription trend (last 30 days)
        $trend = NewsletterSubscriber::select(
            DB::raw('DATE(created_at) as date'),
            DB::raw('COUNT(*) as count')
        )
        ->where('created_at', '>=', now()->subDays(30))
        ->groupBy('date')
        ->orderBy('date', 'asc')
        ->get();

        return response()->json([
            'success' => true,
            'stats' => $stats,
            'trend' => $trend,
        ]);
    }

    /**
     * Unsubscribe from newsletter
     */
    public function unsubscribe(Request $request, $id)
    {
        $subscriber = NewsletterSubscriber::findOrFail($id);

        $subscriber->update([
            'is_active' => false,
            'unsubscribed_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Inscrição cancelada com sucesso.',
        ]);
    }

    /**
     * Delete subscriber (Admin only)
     */
    public function destroy($id)
    {
        $subscriber = NewsletterSubscriber::findOrFail($id);
        $subscriber->delete();

        return response()->json([
            'success' => true,
            'message' => 'Inscrito removido com sucesso.',
        ]);
    }

    /**
     * Bulk export subscribers
     */
    public function export(Request $request)
    {
        $status = $request->get('status', 'active');

        $query = NewsletterSubscriber::query();

        if ($status === 'active') {
            $query->active();
        } elseif ($status === 'inactive') {
            $query->inactive();
        }

        $subscribers = $query->get(['email', 'subscribed_at', 'is_active']);

        return response()->json([
            'success' => true,
            'data' => $subscribers,
        ]);
    }
}
