<?php

namespace App\Http\Controllers;

use App\Models\HeroImage;
use Illuminate\Http\Request;

class HeroController extends Controller
{
    /**
     * Display a listing of the resource (API).
     */
    public function index(Request $request)
    {
        // Se for uma requisição AJAX/API, retorna JSON
        if ($request->wantsJson() || $request->is('api/*')) {
            $heroes = HeroImage::orderBy('order')->where('active', 1)->get();
            return response()->json($heroes);
        }
        
        // Caso contrário, retorna a view admin
        $heroes = HeroImage::orderBy('order')->get();
        return view('admin.hero.index', ['heroes' => $heroes]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'order' => 'required|integer',
            'title_key' => 'required|string',
            'subtitle_key' => 'required|string',
            'description_key' => 'required|string',
            'active' => 'sometimes|boolean',
        ]);

        $validated = [
            'order' => $request->input('order'),
            'title_key' => $request->input('title_key'),
            'subtitle_key' => $request->input('subtitle_key'),
            'description_key' => $request->input('description_key'),
            'active' => $request->boolean('active', true),
        ];

        // Handle file upload
        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            try {
                $file = $request->file('image');
                $filename = 'hero-' . time() . '-' . uniqid() . '.jpg';
                
                // Ensure directory exists
                $path = public_path('images');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }
                
                $file->move($path, $filename);
                $validated['image'] = '/images/' . $filename;
            } catch (\Exception $e) {
                return back()->withErrors(['image' => 'Erro ao fazer upload: ' . $e->getMessage()]);
            }
        } else {
            $validated['image'] = '/images/hero.jpg';
        }

        $validated['active'] = $request->boolean('active', true);
        
        $hero = HeroImage::create($validated);
        
        if ($request->wantsJson()) {
            return response()->json($hero, 201);
        }
        
        return redirect()->route('admin.hero.index')->with('success', 'Slide criado com sucesso');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, HeroImage $heroImage)
    {
        $request->validate([
            'order' => 'sometimes|integer',
            'title_key' => 'sometimes|string',
            'subtitle_key' => 'sometimes|string',
            'description_key' => 'sometimes|string',
            'active' => 'sometimes|boolean',
        ]);

        $validated = [];
        if ($request->has('order')) $validated['order'] = $request->input('order');
        if ($request->has('title_key')) $validated['title_key'] = $request->input('title_key');
        if ($request->has('subtitle_key')) $validated['subtitle_key'] = $request->input('subtitle_key');
        if ($request->has('description_key')) $validated['description_key'] = $request->input('description_key');
        if ($request->has('active')) $validated['active'] = $request->boolean('active');

        // Handle file upload only if a new image is provided
        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            try {
                $file = $request->file('image');
                $filename = 'hero-' . time() . '-' . uniqid() . '.jpg';
                
                // Ensure directory exists
                $path = public_path('images');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }
                
                $file->move($path, $filename);
                $validated['image'] = '/images/' . $filename;
            } catch (\Exception $e) {
                return back()->withErrors(['image' => 'Erro ao fazer upload: ' . $e->getMessage()]);
            }
        }

        if (isset($validated['active'])) {
            $validated['active'] = $request->boolean('active');
        }
        
        $heroImage->update($validated);
        
        if ($request->wantsJson()) {
            return response()->json($heroImage);
        }
        
        return redirect()->route('admin.hero.index')->with('success', 'Slide atualizado com sucesso');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(HeroImage $heroImage)
    {
        $heroImage->delete();
        return redirect()->route('admin.hero.index')->with('success', 'Slide removido com sucesso');
    }
}
