<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\SmsService;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    protected $smsService;

    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Order::with('customer', 'items');
        
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->has('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }
        
        if ($request->has('order_number')) {
            $query->where('order_number', $request->order_number);
        }
        
        $orders = $query->paginate(20);
        
        return response()->json($orders);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Limpar items vazios/inválidos antes de validar
            $items = $request->input('items', []);
            $cleanedItems = array_filter($items, function($item) {
                return isset($item['product_id']) && !empty($item['product_id']) && 
                       isset($item['quantity']) && !empty($item['quantity']) && 
                       isset($item['unit_price']);
            });
            $cleanedItems = array_values($cleanedItems); // Reindexar array

            // Se nenhum item válido, retornar erro
            if (empty($cleanedItems)) {
                return response()->json([
                    'message' => 'No valid items in cart',
                    'errors' => ['items' => ['Cart must have at least one valid item']]
                ], 422);
            }

            // Preparar dados para validação
            $data = $request->all();
            $data['items'] = $cleanedItems;

            // Validação simplificada (sem validar items individual)
            $validator = \Illuminate\Support\Facades\Validator::make($data, [
                'customer_id' => 'required|integer|exists:customers,id',
                'address_id' => 'required|integer|exists:addresses,id',
                'payment_method' => 'required|in:express,reference,transfer',
                'subtotal' => 'required|numeric|min:0',
                'total' => 'required|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();
            $validated['items'] = $cleanedItems;
            $validated['tax'] = $data['tax'] ?? 0;
            $validated['shipping'] = $data['shipping'] ?? 0;
            $validated['status'] = $data['status'] ?? 'pending';

            $orderNumber = 'ORD-' . date('YmdHis') . '-' . rand(100, 999);
            
            // Generate payment reference for transfer and reference methods
            $paymentReference = null;
            if ($validated['payment_method'] === 'transfer' || $validated['payment_method'] === 'reference') {
                $paymentReference = ($validated['payment_method'] === 'transfer' ? 'SOL' : 'REF') . substr(time(), -8);
            }
            
            $order = Order::create([
                'customer_id' => $validated['customer_id'],
                'address_id' => $validated['address_id'],
                'order_number' => $orderNumber,
                'payment_method' => $validated['payment_method'],
                'payment_reference' => $paymentReference,
                'payment_status' => $validated['payment_method'] === 'express' ? 'completed' : 'pending',
                'status' => $validated['status'] ?? 'pending',
                'subtotal' => $validated['subtotal'],
                'tax' => $validated['tax'] ?? 0,
                'shipping' => $validated['shipping'] ?? 0,
                'total' => $validated['total'],
                'notes' => $validated['notes'] ?? null,
            ]);

            foreach ($validated['items'] as $item) {
                $order->items()->create([
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'subtotal' => $item['quantity'] * $item['unit_price'],
                ]);
            }

            // Send SMS notification to customer
            try {
                $customer = $order->load('customer')->customer;
                \Log::info('Attempting to send SMS for order', [
                    'order_id' => $order->id,
                    'order_number' => $orderNumber,
                    'customer_id' => $customer ? $customer->id : null,
                    'customer_phone' => $customer ? $customer->phone : null
                ]);
                
                if ($customer && $customer->phone) {
                    $smsResult = $this->smsService->sendOrderConfirmation(
                        $customer->phone,
                        $orderNumber,
                        $validated['total']
                    );
                    \Log::info('SMS send result', ['success' => $smsResult]);
                } else {
                    \Log::warning('Customer or phone not found for SMS', [
                        'customer' => $customer ? 'exists' : 'null',
                        'phone' => $customer ? $customer->phone : 'N/A'
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('SMS notification failed for order creation', [
                    'order_number' => $orderNumber,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                // Don't fail the order creation if SMS fails
            }

            return response()->json([
                'message' => 'Order created successfully',
                'order' => $order->load('items'),
                'payment_reference' => $paymentReference
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Order validation failed', $e->errors());
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Order creation error', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'message' => 'Error creating order: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        return response()->json($order->load('customer', 'items.product', 'address'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        \Log::info('Order update request received', [
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'current_status' => $order->status,
            'request_data' => $request->all()
        ]);

        $validated = $request->validate([
            'status' => 'in:pending,processing,shipped,delivered,cancelled,refunded',
            'notes' => 'nullable|string',
            'tracking_number' => 'nullable|string',
        ]);

        $oldStatus = $order->status;
        $order->update($validated);

        \Log::info('Order status comparison', [
            'old_status' => $oldStatus,
            'new_status' => $validated['status'] ?? 'not set',
            'status_changed' => isset($validated['status']) && $validated['status'] !== $oldStatus
        ]);

        // Send SMS if status changed
        if (isset($validated['status']) && $validated['status'] !== $oldStatus) {
            try {
                $order->load('customer');
                $customer = $order->customer;
                
                \Log::info('Attempting to send status update SMS', [
                    'order_number' => $order->order_number,
                    'customer_id' => $customer ? $customer->id : null,
                    'customer_phone' => $customer ? $customer->phone : null,
                    'new_status' => $validated['status']
                ]);

                if ($customer && $customer->phone) {
                    $result = $this->smsService->sendOrderStatusUpdate(
                        $customer->phone,
                        $order->order_number,
                        $validated['status']
                    );
                    
                    \Log::info('Status update SMS result', ['success' => $result]);
                } else {
                    \Log::warning('Customer or phone not found for status update SMS', [
                        'customer' => $customer ? 'exists' : 'null',
                        'phone' => $customer ? $customer->phone : 'N/A'
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('SMS notification failed for order update', [
                    'order_number' => $order->order_number,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return response()->json($order);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        $order->delete();

        return response()->json(null, 204);
    }
}
