<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Favorite;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FavoriteController extends Controller
{
    /**
     * List customer favorites
     */
    public function index(Request $request)
    {
        try {
            $customerId = $request->query('customer_id');
            
            if (!$customerId) {
                return response()->json(['error' => 'customer_id is required'], 400);
            }

            // Use raw database query to get products
            $favorites = DB::table('favorites')
                ->where('customer_id', $customerId)
                ->join('products', 'favorites.product_id', '=', 'products.id')
                ->select('products.*')
                ->get();

            return response()->json($favorites, 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        \Log::info('FavoriteController store() called');
        
        try {
            // Accept from both body and query parameters
            $customerId = $request->input('customer_id') ?? $request->query('customer_id');
            $productId = $request->input('product_id') ?? $request->query('product_id');

            \Log::info('Customer ID: ' . $customerId . ', Product ID: ' . $productId);

            if (!$customerId || !$productId) {
                return response()->json(['error' => 'customer_id and product_id are required'], 400);
            }

            // Check if already favorited using raw query
            $exists = DB::table('favorites')
                ->where('customer_id', $customerId)
                ->where('product_id', $productId)
                ->exists();

            if ($exists) {
                return response()->json(['message' => 'Already in favorites'], 200);
            }

            // Insert using raw query
            DB::table('favorites')->insert([
                'customer_id' => $customerId,
                'product_id' => $productId,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return response()->json(['message' => 'Added to favorites'], 201);
        } catch (\Exception $e) {
            \Log::error('FavoriteController store() error: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove product from favorites
     */
    public function destroy(Request $request)
    {
        try {
            // Accept from both body and query parameters
            $customerId = $request->input('customer_id') ?? $request->query('customer_id');
            $productId = $request->input('product_id') ?? $request->query('product_id');

            if (!$customerId || !$productId) {
                return response()->json(['error' => 'customer_id and product_id are required'], 400);
            }

            // Delete using raw query
            $deleted = DB::table('favorites')
                ->where('customer_id', $customerId)
                ->where('product_id', $productId)
                ->delete();

            if (!$deleted) {
                return response()->json(['error' => 'Favorite not found'], 404);
            }

            return response()->json(null, 204);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Check if product is favorited
     */
    public function check(Request $request)
    {
        try {
            $customerId = $request->query('customer_id');
            $productId = $request->query('product_id');

            if (!$customerId || !$productId) {
                return response()->json(['error' => 'customer_id and product_id are required'], 400);
            }

            $favorite = DB::table('favorites')
                ->where('customer_id', $customerId)
                ->where('product_id', $productId)
                ->exists();

            return response()->json(['is_favorite' => $favorite], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
