<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Register a new customer
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'phone' => 'required|string|unique:customers,phone',
            'password' => 'required|string|min:6|confirmed',
        ]);

        try {
            $customer = Customer::create([
                'name' => $validated['first_name'] . ' ' . $validated['last_name'],
                'phone' => $validated['phone'],
                'password' => Hash::make($validated['password']),
                'active' => true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Conta criada com sucesso!',
                'customer' => $customer,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao criar conta: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Login a customer
     */
    public function login(Request $request)
    {
        $validated = $request->validate([
            'phone' => 'required|string',
            'password' => 'required|string',
        ]);

        $customer = Customer::where('phone', $validated['phone'])->first();

        if (!$customer || !Hash::check($validated['password'], $customer->password ?? '')) {
            throw ValidationException::withMessages([
                'phone' => ['Telefone ou senha inválidos.'],
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Login realizado com sucesso!',
            'customer' => $customer,
        ], 200);
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        return response()->json([
            'success' => true,
            'message' => 'Logout realizado com sucesso!',
        ], 200);
    }
}
