<?php

namespace App\Http\Controllers\Api;

use App\Models\Address;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class AddressController
{
    /**
     * Get all addresses for a customer
     */
    public function index($customerId): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($customerId);
            $addresses = $customer->addresses()->orderBy('is_primary', 'desc')->get();
            
            return response()->json($addresses, 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Customer not found',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Store a new address for a customer
     */
    public function store(Request $request, $customerId): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($customerId);
            
            $validated = $request->validate([
                'street' => 'required|string|max:255',
                'city' => 'required|string|max:255',
                'province' => 'required|string|max:255',
                'postal_code' => 'nullable|string|max:20',
                'reference' => 'nullable|string|max:255',
            ]);

            // Se for o primeiro endereço, marca como principal
            $existingAddresses = $customer->addresses()->count();
            $validated['is_primary'] = $existingAddresses === 0;
            $validated['customer_id'] = $customerId;

            $address = Address::create($validated);

            return response()->json([
                'message' => 'Address created successfully',
                'address' => $address
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Address creation error: ' . $e->getMessage());
            return response()->json([
                'message' => 'Error creating address',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete an address
     */
    public function destroy($customerId, $addressId): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($customerId);
            $address = $customer->addresses()->findOrFail($addressId);
            
            // Se for o endereço principal, marca o próximo como principal
            if ($address->is_primary) {
                $nextAddress = $customer->addresses()
                    ->where('id', '!=', $addressId)
                    ->first();
                
                if ($nextAddress) {
                    $nextAddress->update(['is_primary' => true]);
                }
            }

            $address->delete();

            return response()->json([
                'message' => 'Address deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error deleting address',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update an address
     */
    public function update(Request $request, $customerId, $addressId): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($customerId);
            $address = $customer->addresses()->findOrFail($addressId);

            $validated = $request->validate([
                'street' => 'sometimes|required|string|max:255',
                'city' => 'sometimes|required|string|max:255',
                'province' => 'sometimes|required|string|max:255',
                'postal_code' => 'nullable|string|max:20',
                'reference' => 'nullable|string|max:255',
                'is_primary' => 'sometimes|boolean',
            ]);

            // Se marcando como principal, remove de outros
            if (isset($validated['is_primary']) && $validated['is_primary']) {
                $customer->addresses()
                    ->where('id', '!=', $addressId)
                    ->update(['is_primary' => false]);
            }

            $address->update($validated);

            return response()->json([
                'message' => 'Address updated successfully',
                'address' => $address
            ], 200);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error updating address',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a single address
     */
    public function show($customerId, $addressId): JsonResponse
    {
        try {
            $customer = Customer::findOrFail($customerId);
            $address = $customer->addresses()->findOrFail($addressId);

            return response()->json($address, 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Address not found',
                'error' => $e->getMessage()
            ], 404);
        }
    }
}
