<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Customer;
use App\Models\Order;
use App\Models\NewsletterSubscriber;
use App\Models\DealSetting;
use Illuminate\Http\Request;

class AdminController extends Controller
{
    public function dashboard()
    {
        return view('admin.dashboard', [
            'totalProducts' => Product::count(),
            'totalOrders' => Order::count(),
            'totalCustomers' => Customer::count(),
            'totalCategories' => Category::count(),
            'recentOrders' => Order::with('customer')->latest()->take(5)->get(),
        ]);
    }

    // Products
    public function productsIndex()
    {
        return view('admin.products.index', [
            'products' => Product::with('category')->get(),
            'categories' => Category::all(),
        ]);
    }

    public function productsStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'category_id' => 'required|exists:categories,id',
            'price' => 'required|numeric',
            'original_price' => 'nullable|numeric',
            'stock' => 'required|integer',
            'description' => 'nullable|string',
            'active' => 'boolean',
            'is_daily_deal' => 'boolean',
            'is_combo' => 'boolean',
        ]);

        $validated['slug'] = str()->slug($validated['title']);
        
        // Processar upload de imagens
        $imageUrls = [];
        if ($request->hasFile('image_files')) {
            foreach ($request->file('image_files') as $file) {
                $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads'), $filename);
                $imageUrls[] = '/uploads/' . $filename;
            }
        }
        
        $validated['images'] = $imageUrls;
        Product::create($validated);

        return redirect()->route('admin.products.index')->with('success', 'Produto criado com sucesso!');
    }

    public function productsDestroy(Product $product)
    {
        $product->delete();
        return redirect()->route('admin.products.index')->with('success', 'Produto removido com sucesso!');
    }

    public function productsUpdate(Request $request, Product $product)
    {
        $validated = $request->validate([
            'title' => 'sometimes|required|string',
            'category_id' => 'sometimes|required|exists:categories,id',
            'price' => 'sometimes|required|numeric',
            'original_price' => 'nullable|numeric',
            'stock' => 'sometimes|required|integer',
            'description' => 'nullable|string',
            'active' => 'boolean',
            'is_daily_deal' => 'boolean',
            'is_combo' => 'boolean',
        ]);

        if (isset($validated['title'])) {
            $validated['slug'] = str()->slug($validated['title']);
        }
        
        // Remover imagens que foram marcadas para remoção
        if ($request->has('images_to_remove')) {
            $imageUrls = $product->images ?? [];
            $imagesToRemove = $request->input('images_to_remove');
            $imageUrls = array_filter($imageUrls, function($img) use ($imagesToRemove) {
                return !in_array($img, $imagesToRemove);
            });
            $imageUrls = array_values($imageUrls); // Re-indexar array
            
            // Deletar arquivos do disco
            foreach ($imagesToRemove as $imagePath) {
                $filePath = public_path($imagePath);
                if (file_exists($filePath)) {
                    unlink($filePath);
                }
            }
            
            $validated['images'] = $imageUrls;
        }
        
        // Processar upload de imagens novas
        if ($request->hasFile('image_files')) {
            $imageUrls = $product->images ?? [];
            foreach ($request->file('image_files') as $file) {
                $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads'), $filename);
                $imageUrls[] = '/uploads/' . $filename;
            }
            
            $validated['images'] = $imageUrls;
        }
        
        $product->update($validated);

        // Se for uma requisição AJAX, retornar JSON
        if ($request->wantsJson()) {
            return response()->json($product);
        }

        return redirect()->route('admin.products.index')->with('success', 'Produto atualizado com sucesso!');
    }

    // Categories
    public function categoriesIndex()
    {
        return view('admin.categories.index', [
            'categories' => Category::withCount('products')->get(),
        ]);
    }

    public function categoriesStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string',
            'description' => 'nullable|string',
        ]);

        $validated['slug'] = str()->slug($validated['name']);
        Category::create($validated);

        return redirect()->route('admin.categories.index')->with('success', 'Categoria criada com sucesso!');
    }

    public function categoriesDestroy(Category $category)
    {
        $category->delete();
        return redirect()->route('admin.categories.index')->with('success', 'Categoria removida com sucesso!');
    }

    // Customers
    public function customersIndex()
    {
        return view('admin.customers.index', [
            'customers' => Customer::all(),
        ]);
    }

    // Orders
    public function ordersIndex()
    {
        return view('admin.orders.index', [
            'orders' => Order::with('customer')->latest()->get(),
        ]);
    }

    public function ordersShow(Order $order)
    {
        return view('admin.orders.show', [
            'order' => $order->load('customer', 'items'),
        ]);
    }

    public function ordersUpdate(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,processing,shipped,delivered,cancelled',
        ]);

        $order->update($validated);

        return redirect()->route('admin.orders.show', $order)->with('success', 'Status da encomenda atualizado com sucesso!');
    }

    // Newsletter
    public function newsletterIndex()
    {
        return view('admin.newsletter.index', [
            'subscribers' => NewsletterSubscriber::orderBy('created_at', 'desc')->paginate(50),
            'stats' => [
                'total' => NewsletterSubscriber::count(),
                'active' => NewsletterSubscriber::where('is_active', true)->count(),
                'inactive' => NewsletterSubscriber::where('is_active', false)->count(),
                'today' => NewsletterSubscriber::whereDate('created_at', today())->count(),
            ]
        ]);
    }

    public function newsletterDestroy(NewsletterSubscriber $subscriber)
    {
        $subscriber->delete();
        return redirect()->route('admin.newsletter.index')->with('success', 'Subscritor removido com sucesso!');
    }

    // Deal Settings
    public function dealSettingsIndex()
    {
        $setting = DealSetting::where('key', 'daily_deals_end_time')->first();
        
        return view('admin.deals.index', [
            'setting' => $setting,
            'remaining_hours' => $setting ? max(0, now()->diffInHours($setting->end_time, false)) : 24,
        ]);
    }

    public function dealSettingsUpdate(Request $request)
    {
        $validated = $request->validate([
            'hours' => 'required|integer|min:1|max:168',
        ]);

        DealSetting::updateOrCreate(
            ['key' => 'daily_deals_end_time'],
            [
                'end_time' => now()->addHours((int) $validated['hours']),
                'is_active' => true,
            ]
        );

        return redirect()->route('admin.deals.index')->with('success', 'Tempo de oferta atualizado com sucesso!');
    }
}
